<?php


/**
 * Class CombatModel. Modelo de la clase CombatModel
 */
class CombatModel extends CI_Model
{
    const TABLE_NAME = 'Combate';

    /**
     * CombatModel constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    /**
     * @param $date Fecha en la que se realiza la competicion
     * @param $idWeight Id unico del peso del competidor
     * @return Persona [] . Que estan registradas en el peso indicado por parametro en la fecha indicada tambien por parametro
     */
    public function getCompetitorsByCompetitionAndIdWeight($date,$idWeight) {
        $sql = 'SELECT P.*
                FROM Persona P , Participacion PA 
                WHERE P.dni = PA.dniCompetidor AND PA.fecha_participacion = ? AND PA.idPeso = ?';

        $query = $this->db->query($sql, array($date, $idWeight));

        return $query -> result();


    }

    /**
     * Funcion que crea un combate
     * @param $idMat Id del tatami al que se le va a asignar el combate
     * @param $phase Fase en la que se va a desarrollar el encuentro
     * @trow Exception si existe error a la hora de insertar en la BD.
     */
    public function create($idMat,$phase) {
        $combat = array(
            'fase' => $phase,
            'idTatami' => $idMat
        );
        $this->db->trans_start();
        $this->db->insert(self::TABLE_NAME, $combat);
        $idCombat = $this->db->insert_id(); // returns the latest id create by mysqli


        $this->db->trans_complete();
        if (!$this->db->trans_status()) throw new Exception('Error, no se pudo crear el combate');
        return $idCombat;
    }

    /**
     * @param $idCombat id del combate al que se le quiere asociar la puntuacion (Score)
     * @param $dniCompetitor identificador del competidor
     * @param $color_kimono color del kimono ( B | A)
     * @return Id del combate
     * @throws Exception Si existe algun error a la hora de insertar en la BD
     */
    public function createScore($idCombat,$dniCompetitor,$color_kimono) {
        $score = array(
            'idCombate' => $idCombat,
            'dniCompetidor' => $dniCompetitor,
            'color_kimono' => $color_kimono
        );
        $this->db->trans_start();
        $this->db->insert('Puntuacion', $score);
        $idCombat = $this->db->insert_id(); // returns the latest id create by mysqli


        $this->db->trans_complete();
        if (!$this->db->trans_status()) throw new Exception('Error, no se pudo crear la puntacion');
        return $idCombat;
    }

    /**
     * Funcion que retorna los combates de una determinada competicion y peso
     * @param $idCompetition Identificador de la competicion
     * @param $date Fecha en la que se realiza la competicion / inscripcion
     * @param $idWeight Identificador del peso
     * @return mixed Combate []  Personas con sus correspondientes puntuaciones
     */
    public function getCombats($idCompetition,$date,$idWeight) {
        $sql  ='SELECT P1.ippon as ippon_b, P1.waza_ari as waza_ari_b, P1.shido as shido_b, P1.hansokumake as hansokumake_b,
                P2.ippon as ippon_w, P2.waza_ari as waza_ari_w, P2.shido as shido_w, P2.hansokumake as hansokumake_w ,
                PE1.nombre as nombre_b,PE1.apellido1 as apellido1_b, PE1.apellido2 as apellido2_b,PE1.dni as dni_b,
                PE2.nombre as nombre_w,PE2.apellido1 as apellido1_w, PE2.apellido2 as apellido2_w, PE2.dni as dni_w,
                C.*
                FROM Puntuacion P1, Puntuacion P2,Persona PE1, Persona PE2, Combate C
                WHERE P1.idCombate IN (SELECT C.idCombate
                                        FROM Combate C
                                        WHERE C.idTatami IN ( SELECT T.idTatami
                                                              FROM Tatami T
                                                              WHERE T.idCompeticion = ?))
                AND P1.dniCompetidor IN (SELECT PA.dniCompetidor
                                          FROM Participacion PA
                                          WHERE PA.fecha_participacion = ? and PA.idPeso = ?)
                AND  P2.idCombate IN (SELECT C.idCombate
                                        FROM Combate C
                                        WHERE C.idTatami IN ( SELECT T.idTatami
                                                              FROM Tatami T
                                                              WHERE T.idCompeticion = ?))
                AND P2.dniCompetidor IN (SELECT PA.dniCompetidor
                                          FROM Participacion PA
                                          WHERE PA.fecha_participacion = ? and PA.idPeso = ?)
                AND P1.dniCompetidor <> P2.dniCompetidor and P1.idCombate = P2.idCombate
                
                AND PE1.dni = P1.dniCompetidor AND PE2.dni = P2.dniCompetidor AND C.idCombate = P1.idCombate AND P2.idCombate= C.idCombate
                AND P1.color_kimono = ? AND P2.color_kimono = ?
                
                group by C.idCombate;
                
               ';

        $query = $this -> db -> query ($sql,array($idCompetition,$date,$idWeight,$idCompetition,$date,$idWeight,'A','B'));
        return $query -> result() ;


    }

    /**
     * Funcion que retorna
     * @param $idMat Identificador de tatami
     * @return mixed Combate []  Personas con sus correspondientes puntuaciones
     */
    public function getCombatsByIDMat($idMat) {
        $sql = 'SELECT P1.ippon as ippon_b, P1.waza_ari as waza_ari_b, P1.shido as shido_b, P1.hansokumake as hansokumake_b,
                P2.ippon as ippon_w, P2.waza_ari as waza_ari_w, P2.shido as shido_w, P2.hansokumake as hansokumake_w ,
                PE.nombre as nombre_b,PE.apellido1 as apellido1_b, PE.apellido2 as apellido2_b,PE.dni as dni_b,
                PE2.nombre as nombre_w,PE2.apellido1 as apellido1_w, PE2.apellido2 as apellido2_w, PE2.dni as dni_w,
                C.*
                FROM Puntuacion P1,Puntuacion P2, Persona PE, Persona PE2, Combate C
                WHERE C.idTatami = ? 
                      AND P1.idCombate = C.idCombate
                      AND P2.idCombate = C.idCombate
                      AND P1.dniCompetidor <> P2.dniCompetidor
                      AND PE.dni = P1.dniCompetidor
                      AND PE2.dni = P2.dniCompetidor
                      AND P1.color_kimono = ?
                      AND P2.color_kimono = ?
                group by C.idCombate';

        $query = $this -> db -> query($sql,array($idMat,'A','B'));
        return $query -> result();
    }

    /**
     * Funcion para añadir una puntuacion a un combate
     * @param $dni identificador del competidor
     * @param $ippon Puntuacion de tipo Ippon
     * @param $wazari Puntuacion de tipo Wazaari
     * @param $shido Puntuacion de tipo Shido
     * @param $hansoku Puntuacion de tipo Hansoku
     * @param $idCombat Identificador del combate
     * @throws Exception si existe error a la hora de insertar en la BD
     */
    public function addScore($dni,$ippon,$wazari,$shido,$hansoku, $idCombat) {

        $score = array(
            'ippon' => $ippon,
            'waza_ari' => $wazari,
            'shido' => $shido,
            'hansokumake' => $hansoku
        );

        $this->db->trans_start();
        // Insertamos la puntuacion del competidor
        $this -> db -> where ('idCombate',$idCombat);
        $this -> db -> where ('dniCompetidor',$dni);

        $this -> db -> update('Puntuacion',$score);
        $this->db->trans_complete();
        if (!$this->db->trans_status()) throw new Exception('Error, no se pudo añadir la puntuacion');

    }

    /**
     * Funcion para actualizar el tatami de un determinado combate
     * @param $idCombat Identificador del tatami que se quiere realizar el cambio de tatami
     * @param $idMat Identificador del nuevo tatami que se va a cambiar
     * @throws Exception Si existe error a la hora de realizar el cambio de tatami
     */
    public function updateMat($idCombat,$idMat) {
        $combat = array(
            'idTatami' => $idMat
        );
        $this->db->trans_start();
        $this -> db -> where ('idCombate ',$idCombat);
        $this -> db -> update('Combate',$combat);
        $this->db->trans_complete();
        if (!$this->db->trans_status()) throw new Exception('Error, no se pudo actualizar el numero de tatami 
        asociado al combate.');
    }

    /**
     * @param $dni Identificador del competidor
     * @param $idWeight Identificador del peso
     * @param $position posicion en la que ha quedado ( x en N)
     * @param $date Fecha en la que se ha realizado la competicion
     * @throws Exception
     */
    public function enterPosition($dni,$idWeight,$position,$date) {
        $participation = array(
            'puesto' => $position
        );
        $this->db->trans_start();
        $this -> db -> where ('dniCompetidor ',$dni);
        $this -> db -> where ('idPeso ',$idWeight);
        $this -> db -> where  ('fecha_participacion',$date);
        $this -> db -> update('Participacion',$participation);
        $this->db->trans_complete();
        if (!$this->db->trans_status()) throw new Exception('Error, no se pudo añadir la puntuacion al competidor indicado.');
    }

    /**
     * Funcion para eliminar un determinado combate cuyo identificador es pasado por parametro
     * @param $idCombat Identificador del combate
     * @throws Exception Si ocurre un error a la hora de eliminar el combate
     */
    public function delete($idCombat) {

        $this->db->where('idCombate', $idCombat);
        $this->db->trans_start();
        $this->db->delete('Combate');
        $this->db->trans_complete();
        if (!$this->db->trans_status()) {
            throw new Exception('Error, it was impossible delete that combat');
        }
    }

    /**
     * Funcion que devuelve los arbitros disponibles para un determinado combate. Pueden ser aquellos arbitros que esten inscritos en la competicion
     * y asignados al tatami donde se desarrollara el combate. Puede arbitrar si y solo si ninguno de los dos competidores es de su mismo club
     * @param $dniW Identificador del competidor de kimono blanco
     * @param $dniB Identificador del comepetidor de kimono azul
     * @param $idCompetition Identificador de la competicion
     * @param $idMat Identificador del tatami
     * @return mixed Persona [] con el nombre del equipo
     */
    public function getReferees($dniW,$dniB,$idCompetition,$idMat) {
        $sql = 'SELECT A.nombre as nombreArbitro, A.* ,E.nombre
FROM Persona A ,arbitroatatami T,inscripcion I ,Equipo E
WHERE T.idTatami = ? AND T.dniArbitro = A.dni AND I.idCompeticion = ? and I.dniPersona = A.dni and E.nombre = I.nombreEquipo

and I.nombreEquipo <> (SELECT E.nombre
                          FROM Equipo E, Inscripcion I 
                          WHERE E.nombre = I.nombreEquipo and I.dniPersona = ? and I.idCompeticion = ?)
and I.nombreEquipo <> (SELECT E.nombre
                          FROM Equipo E, Inscripcion I 
                          WHERE E.nombre = I.nombreEquipo and I.dniPersona = ? and I.idCompeticion = ?)';

        $query = $this -> db -> query($sql,array($idMat,$idCompetition,$dniW,$idCompetition,$dniB,$idCompetition));
        return $query -> result();
    }



}